from pydantic import BaseModel, Field
from datetime import datetime
from typing import Optional, List, Dict

# -------------------------------
# Comment Model – for team status updates
# -------------------------------
class Comment(BaseModel):
    id: Optional[str] = Field(None, alias="_id")
    account_id: str
    team_member_id: str
    comment: str
    created_date: datetime = Field(default_factory=datetime.utcnow)

# -------------------------------
# SalesStatus Model – a single status entry in the sales pipeline
# -------------------------------
class SalesStatus(BaseModel):
    status: str  # e.g., "Lead", "Contacted", "Negotiation", "Won", "Lost"
    updated_date: datetime = Field(default_factory=datetime.utcnow)

# -------------------------------
# AccountDetail Model – stores extended CRM details for an account
# -------------------------------
class AccountDetail(BaseModel):
    id: Optional[str] = Field(None, alias="_id")
    account_id: str
    description: Optional[str] = None
    sales_pipeline: Optional[List[SalesStatus]] = []  # Allow up to 5 statuses
    lead_information: Optional[Dict[str, str]] = None  # e.g., {"source": "web", "campaign": "summer"}
    created_date: datetime = Field(default_factory=datetime.utcnow)
    updated_date: datetime = Field(default_factory=datetime.utcnow)

# -------------------------------
# AccountStats Model – aggregate statistics for an account
# -------------------------------
class AccountStats(BaseModel):
    total_comments: int
    last_comment_date: Optional[datetime]
    total_users: int
    monthly_payments: Optional[float] = 0.0
    current_sales_status: Optional[str] = None

# -------------------------------
# Log Model – for activity logs
# -------------------------------
class Log(BaseModel):
    id: Optional[str] = Field(None, alias="_id")
    account_id: str
    activity: str
    details: Optional[str] = None
    timestamp: datetime = Field(default_factory=datetime.utcnow)
